# -*- coding: utf-8 -*-
import csv
import mysql.connector
from mysql.connector import Error
from pathlib import Path
import sys
import re
import traceback

def process_primary_table(connection, csv_file_path_str, table_name):
    """
    Apstrādā galveno 'out-all.csv' failu, izmantojot ĀTRU pakešu ievietošanu.
    """
    print(f"\n--- SĀKU APSTRĀDĀT PRIMĀRO TABULU: {csv_file_path_str} -> {table_name} ---")
    
    csv_file_path = Path(csv_file_path_str)
    # Specifiski kolonu nosaukumi primārajai tabulai
    csv_col_eka = 'eka'
    csv_col_point = 'point'
    csv_col_cilveki = 'cilveki'
    csv_col_level = 'level'
    db_col_eka = 'eka'
    db_col_point = 'location'
    db_col_cilveki = 'cilveki'
    db_col_level = 'level'

    cursor = None
    try:
        if not csv_file_path.is_file():
            raise FileNotFoundError(f"Primārais CSV fails '{csv_file_path}' nav atrasts!")
        print(f"Primārais CSV fails '{csv_file_path}' atrasts.")

        cursor = connection.cursor()

        create_table_sql = f"""
            CREATE TABLE IF NOT EXISTS `{table_name}` (
                `{db_col_eka}` VARCHAR(20) NOT NULL,
                `{db_col_cilveki}` VARCHAR(50) NULL,
                `{db_col_level}` VARCHAR(10) NULL,
                `{db_col_point}` POINT NOT NULL,
                SPATIAL INDEX(`{db_col_point}`),
                PRIMARY KEY (`{db_col_eka}`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        """
        cursor.execute(create_table_sql)
        cursor.execute(f"TRUNCATE TABLE `{table_name}`;")
        connection.commit()
        print(f"Tabula '{table_name}' ir notīrīta un gatava.")

        insert_sql = f"""
            INSERT INTO `{table_name}` (`{db_col_eka}`, `{db_col_cilveki}`, `{db_col_level}`, `{db_col_point}`)
            VALUES (%s, %s, %s, ST_PointFromText(%s, 4326))
        """

        print(f"Sākam datu importu no '{csv_file_path}'...")
        row_count = 0
        error_count = 0
        inserted_count = 0
        
        # --- ĀTRUMA UZLABOJUMI ---
        data_batch = []
        batch_size = 1000 # Cik rindas krāt pirms sūtīšanas uz DB
        # -------------------------

        with open(csv_file_path, mode='r', encoding='utf-8') as csv_file:
            csv_reader = csv.DictReader(csv_file, delimiter=',')
            for row in csv_reader:
                row_count += 1
                try:
                    eka_val = row[csv_col_eka]
                    point_wkt_original = row[csv_col_point]
                    cilveki_val = row[csv_col_cilveki]
                    level_val = row[csv_col_level]

                    if not point_wkt_original or not point_wkt_original.upper().startswith("POINT(") or not eka_val:
                        error_count += 1
                        continue

                    match = re.match(r'POINT\(\s*(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*\)', point_wkt_original.strip(), re.IGNORECASE)
                    if not match:
                        error_count += 1
                        continue
                    
                    # Šeit saglabājam oriģinālo loģiku primārajai tabulai (ja tā strādāja pareizi)
                    lat_str, lon_str = match.group(1), match.group(2)
                    corrected_point_wkt = f"POINT({lon_str} {lat_str})"
                    
                    cilveki_db = cilveki_val if cilveki_val else None
                    level_db = level_val if level_val else None
                    
                    # --- ĀTRUMA UZLABOJUMI ---
                    # Pievienojam rindu paketei, nevis izpildām uzreiz
                    data_batch.append((eka_val, cilveki_db, level_db, corrected_point_wkt))
                    
                    # Ja pakete ir pilna, ievietojam datus un iztīram sarakstu
                    if len(data_batch) >= batch_size:
                        cursor.executemany(insert_sql, data_batch)
                        connection.commit()
                        inserted_count += len(data_batch)
                        print(f"  ...ievietota pakete, kopā {inserted_count} rindas no {csv_file_path.name}...")
                        data_batch = [] # Iztīrām paketi
                    # -------------------------
                except Exception:
                    error_count += 1
        
        # --- ĀTRUMA UZLABOJUMI ---
        # Ievietojam atlikušās rindas, kas palika pēdējā nepilnajā paketē
        if data_batch:
            cursor.executemany(insert_sql, data_batch)
            connection.commit()
            inserted_count += len(data_batch)
            print(f"  ...ievietota pēdējā pakete, kopā {inserted_count} rindas.")
        # -------------------------
        
        print(f"Datu importēšana pabeigta failam {csv_file_path.name}.")
        print(f"Kopā apstrādātas: {row_count}, veiksmīgi ievietotas: {inserted_count}, izlaistas/ar kļūdām: {error_count}")

    except Exception as e:
        print(f"KĻŪDA, apstrādājot primāro tabulu '{table_name}': {e}", file=sys.stderr)
        traceback.print_exc(file=sys.stderr)
    finally:
        if cursor:
            cursor.close()
        print(f"--- PABEIGTA PRIMĀRĀS TABULAS APSTRĀDE ---")


def process_secondary_table(connection, csv_file_path_str, table_name):
    """
    Apstrādā sekundāros CSV failus, izmantojot ĀTRU pakešu ievietošanu.
    """
    print(f"\n--- Sāku apstrādāt: {csv_file_path_str} -> {table_name} ---")

    csv_file_path = Path(csv_file_path_str)
    csv_col_point = 'point'
    csv_col_name = 'name'
    db_col_name = 'name'
    db_col_point = 'location'
    db_col_id = 'id'

    cursor = None
    try:
        if not csv_file_path.is_file():
            raise FileNotFoundError(f"CSV fails '{csv_file_path}' nav atrasts!")
        print("CSV fails atrasts.")

        cursor = connection.cursor()

        create_table_sql = f"""
            CREATE TABLE IF NOT EXISTS `{table_name}` (
                `{db_col_id}` INT AUTO_INCREMENT PRIMARY KEY,
                `{db_col_name}` VARCHAR(255) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci NULL,
                `{db_col_point}` POINT NOT NULL,
                SPATIAL INDEX(`{db_col_point}`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        """
        cursor.execute(create_table_sql)
        cursor.execute(f"TRUNCATE TABLE `{table_name}`;")
        connection.commit()
        print(f"Tabula '{table_name}' ir notīrīta un gatava.")

        insert_sql = f"""
            INSERT INTO `{table_name}` (`{db_col_name}`, `{db_col_point}`)
            VALUES (%s, ST_PointFromText(%s, 4326))
        """

        print(f"Sākam datu importu no '{csv_file_path}'...")
        row_count, error_count, inserted_count = 0, 0, 0

        # --- ĀTRUMA UZLABOJUMI ---
        data_batch = []
        batch_size = 1000 # Cik rindas krāt pirms sūtīšanas uz DB
        # -------------------------

        with open(csv_file_path, mode='r', encoding='utf-8') as csv_file:
            csv_reader = csv.DictReader(csv_file, delimiter=',', quoting=csv.QUOTE_MINIMAL)
            for row in csv_reader:
                row_count += 1
                try:
                    name_val = row.get(csv_col_name)
                    point_wkt_original = row.get(csv_col_point)

                    if not point_wkt_original or not point_wkt_original.upper().startswith("POINT("):
                        error_count += 1
                        continue
                    
                    match = re.match(r'POINT\(\s*(-?\d+\.?\d*)\s+(-?\d+\.?\d*)\s*\)', point_wkt_original.strip(), re.IGNORECASE)
                    if not match:
                        error_count += 1
                        continue

                    # --- LABOJUMS SĀKAS ŠEIT ---
                    # 1. iestādes.py ģenerē failus formātā POINT(lon lat).
                    # MySQL sagaida formātu POINT(lon lat).
                    # Tāpēc mēs datus NEMAINĀM vietām, bet ņemam tieši tādā secībā, kādā tie ir CSV.
                    
                    lon_str, lat_str = match.group(1), match.group(2) # Oriģināli šeit bija kļūdaina lat/lon piešķiršana
                    corrected_point_wkt = f"POINT({lon_str} {lat_str})"
                    # --- LABOJUMS BEIDZAS ŠEIT ---
                    
                    name_db = name_val.strip() if name_val else None

                    # --- ĀTRUMA UZLABOJUMI ---
                    data_batch.append((name_db, corrected_point_wkt))
                    
                    if len(data_batch) >= batch_size:
                        cursor.executemany(insert_sql, data_batch)
                        connection.commit()
                        inserted_count += len(data_batch)
                        print(f"  ...ievietota pakete, kopā {inserted_count} rindas no {csv_file_path.name}...")
                        data_batch = []
                    # -------------------------
                
                except Exception:
                    error_count += 1

        # --- ĀTRUMA UZLABOJUMI ---
        # Ievietojam atlikušās rindas
        if data_batch:
            cursor.executemany(insert_sql, data_batch)
            connection.commit()
            inserted_count += len(data_batch)
            print(f"  ...ievietota pēdējā pakete, kopā {inserted_count} rindas.")
        # -------------------------

        print(f"\nDatu importēšana pabeigta failam {csv_file_path.name}.")
        print(f"Kopā apstrādātas: {row_count}, veiksmīgi ievietotas: {inserted_count}, izlaistas/ar kļūdām: {error_count}")

    except Exception as e:
        print(f"KĻŪDA, apstrādājot '{table_name}': {e}", file=sys.stderr)
        traceback.print_exc(file=sys.stderr)
    finally:
        if cursor:
            cursor.close()
        print(f"--- Pabeigta apstrāde: {csv_file_path_str} -> {table_name} ---")


def main():
    print("Sāku Python skripta izpildi...")

    # MySQL pieslēgšanās dati
    db_host = "xxx"
    db_port = xxxx
    db_name = "xxx"
    db_user = "xxx"
    db_pass = "xxx"

    # Fails -> Tabula mapējums sekundārajiem failiem
    secondary_files_to_tables = {
        "cofe.csv": "u260470642_cofe",
        "bar.csv": "u260470642_bar",
        "food.csv": "u260470642_food",
        "frizieri.csv": "u260470642_frizieri"
    }

    connection = None
    try:
        print(f"\nPieslēdzos MySQL serverim ({db_host}:{db_port}), datubāze: {db_name}")
        connection = mysql.connector.connect(
            host=db_host, port=db_port, database=db_name, user=db_user, password=db_pass,
            charset='utf8mb4', collation='utf8mb4_unicode_ci'
        )

        if connection.is_connected():
            print("Globālais savienojums veiksmīgs!\n")

            # 1. SOLIS: Apstrādājam primāro tabulu
            process_primary_table(connection, "out-all.csv", "buildings_geo")

            # 2. SOLIS: Apstrādājam visas sekundārās tabulas
            for csv_file, table_name in secondary_files_to_tables.items():
                process_secondary_table(connection, csv_file, table_name)
        else:
            print("Neizdevās izveidot savienojumu ar MySQL serveri.", file=sys.stderr)

    except Error as e:
        print(f"Kļūda, pieslēdzoties vai darbojoties ar MySQL: {e}", file=sys.stderr)
        sys.exit(1)
    except Exception as e_main:
         print(f"Neparedzēta KĻŪDA galvenajā skripta daļā: {e_main}", file=sys.stderr)
         traceback.print_exc(file=sys.stderr)
         sys.exit(1)
    finally:
        if connection and connection.is_connected():
            connection.close()
            print("\nGlobālais savienojums ar MySQL ir aizvērts.")

    print("\nVISU FAILU APSTRĀDE PABEIGTA.")

if __name__ == '__main__':
    main()